using System;
using System.Reflection;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.HL7.Parsers
{

	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian Tomlin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>3/28/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// CPRS HL7 message parser
	/// </summary>
	public class CprsHL7Parser
	{
		private const string CPRS_CLINICAL_ORDER_MESSAGE = "OMG^O19";
		private const string RESPONSE_TO_CPRS_CLINICAL_ORDER_MESSAGE = "ORG^O20";

        // Logger
        private static readonly ILogger _logger =
            LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);
        // Events Logger
        private static readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5566"> 
		///		<ExpectedInput>Valid HL7 Message New Order message for a TAS</ExpectedInput>
		///		<ExpectedOutput>non-null HL7ProtocolMessage containing valid HL7 Acknowledgement Message</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5567"> 
		///		<ExpectedInput>null messageString</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5930"> 
		///		<ExpectedInput>null HL7Interface input parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5939"> 
		///		<ExpectedInput>Invalid HL7Interface input parameter. Patient Update used instead of CPRS.</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6970"> 
		///		<ExpectedInput>Valid HL7Interface, but unsupported Message Type for CPRS.</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Primary method to identify 
		/// </summary>
		/// <param name="intParms"><see cref="HL7Interface"/> containing interface parameters for the CPRS interface.</param>
		/// <param name="messageString">HL7 message string.</param>
		/// <returns></returns>
		public static HL7ProtocolMessage ParseHL7Message( HL7Interface intParms, string messageString )
		{
			if( intParms == null )
				throw( new ArgumentNullException( "intParms" ) );

			if( intParms.InterfaceName != InterfaceName.CPRS.ToString() )
				throw( new HL7Exception( "CprsHL7Parser.ParseHL7Message can only be used with the CPRS HL7Interface" ) );

			if( messageString == null || messageString == string.Empty )
				throw( new ArgumentNullException( "messageString" ) );

			HL7ProtocolMessage _ackMessage = null;

			if ( HL7Utility.ParseGetHL7MessageTypeAndTriggerEventString( messageString ) == CPRS_CLINICAL_ORDER_MESSAGE )
			{
				HL7OmgMessage _omgMessage = new HL7OmgMessage( messageString );
				try
				{
					_ackMessage = ProcessOmgMessage( intParms, _omgMessage );
				}
                // *** Fortify Justified Code ***
                // *** We dont suppress messages and use _ackMessage inside try/catch to maintain errored objects state. Outer functionality is designed to use that property to check if error occurred and handle it appropriately.***
                catch (Exception exc)
                {
                    var messageBody = "Exception Message:\n" + HL7Utility.GetInnerExceptionFromException(exc);
                    // EventLogAppender
                    _eventsLogger.Error("CPRS HL7 Parser: " + messageBody);
                    // Duplicate error information in general logger
                    _logger.Error("CPRS HL7 Parser: " + messageBody);

                    _ackMessage = CprsOrgMessage.AckError(_omgMessage, AckCodes.AR, HL7Utility.GetInnerExceptionFromException(exc));
                    InsertUpdateMessageLog(_omgMessage, _ackMessage, MessageStatus.ProcessingError, intParms, HL7Utility.GetInnerExceptionFromException(exc));
                    //TODO: dkozlov - Email can be controlled by log4net if necessary.
                    // CR 1928 - BNT 6/2/06 Removed HL7 Message text from email message.
                    var mail = new HL7MailMessage();
                    mail.SendMessage(intParms.InterfaceAdministratorEmail, PII                         ", "CPRS HL7 Parser caught System.Exception", messageBody, "smtp.DNS   ");
                    return _ackMessage;
                }
                // *** Fortify Justified Code *** 

            }
			else
			{
				throw( new HL7Exception("Unsupported Message Type of (" + HL7Utility.ParseGetHL7MessageTypeAndTriggerEventString( messageString ) +
					") was received for the CPRS Interface.\n\n" + "HL7 Message:\n" + messageString));
			}
			return _ackMessage;
		}

		private static HL7ProtocolMessage ProcessOmgMessage( HL7Interface intParms, HL7OmgMessage message )
		{
			if ( message == null )
				throw( new ArgumentNullException( "message" ) );

			HL7ProtocolMessage _ackMessage = null;
			string errorText = null;
			HL7MessageLog.InsertMessageLog( HL7MessageLog.GetMessageDataForMessageLogInsert( message.GetMessage(), MessageStatus.ProcessingIncomingMessage, intParms, Common.UpdateFunction.HL7CprsInterface ) );

			if ( message.OrderControlCode == "NW" )
			{
				_ackMessage = new CprsNewOrder( message, intParms ).ProcessNewOrder();
				InsertUpdateMessageLog( message, _ackMessage, MessageStatus.SuccessfullyCompleted, intParms, errorText );
			}
			if ( message.OrderControlCode == "CA" )
			{
				_ackMessage = new CprsCancelVbecsOrder( message ).ProcessCancelOrder();
			}
			if ( message.OrderControlCode == "PR" )
			{
				_ackMessage = new CprsProcessPreviousLabResults( message ).ProcessPreviousResults();
				InsertUpdateMessageLog( message, _ackMessage, MessageStatus.SuccessfullyCompleted, intParms, errorText );
			}
			return _ackMessage;
		}

		private static void InsertUpdateMessageLog( HL7ProtocolMessage incomingMessage, HL7ProtocolMessage ackMessage, MessageStatus status, HL7Interface intParms, string errorText )
		{

			HL7MessageLog.InsertMessageLog( HL7MessageLog.GetMessageDataForMessageLogInsert( incomingMessage.GetMessage(), status, intParms, Common.UpdateFunction.HL7CprsInterface ) );
			if(ackMessage != null)
			{
				HL7MessageLog.InsertAckMessageControlId( incomingMessage.GetMessageControlID(), ackMessage.GetMessageControlID(), status, intParms.InterfaceName, Common.UpdateFunction.HL7CprsInterface );
				
				// CR 2961
				HL7MessageLog.UpdateMessageStatus( status, incomingMessage.GetMessageControlID(), errorText, 0, intParms.InterfaceName, Common.UpdateFunction.HL7CprsInterface );
			}
		}
	}
}
